/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ccvs.ui.wizards;


import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.team.core.RepositoryProvider;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.ccvs.core.CVSProviderPlugin;
import org.eclipse.team.internal.ccvs.core.CVSTeamProvider;
import org.eclipse.team.internal.ccvs.core.client.Command.LocalOption;
import org.eclipse.team.internal.ccvs.ui.CVSUIPlugin;
import org.eclipse.team.internal.ccvs.ui.Policy;

/**
 * An operation to run the CVS diff operation on a set of resources. The result
 * of the diff is written to a file. If there are no differences found, the
 * user is notified and the output file is not created.
 */
public class GenerateDiffFileOperation implements IRunnableWithProgress {

	private File outputFile;
	private IResource resource;
	private Shell shell;
	private LocalOption[] options;
	private boolean toClipboard;

	GenerateDiffFileOperation(IResource resource, File file, boolean toClipboard, LocalOption[] options, Shell shell) {
		this.resource = resource;
		this.outputFile = file;
		this.shell = shell;
		this.options = options;
		this.toClipboard = toClipboard;
	}

	/**
	 * @see IRunnableWithProgress#run(IProgressMonitor)
	 */
	public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
		try {
			monitor.beginTask("", 500); //$NON-NLS-1$
			monitor.setTaskName(
				Policy.bind("GenerateCVSDiff.working")); //$NON-NLS-1$
			
			OutputStream os;
			if(toClipboard) {
				os = new ByteArrayOutputStream();
			} else {
				os = new FileOutputStream(outputFile);
			}
			try {
				CVSTeamProvider provider = (CVSTeamProvider)RepositoryProvider.getProvider(resource.getProject(), CVSProviderPlugin.getTypeId());
				provider.diff(resource, options, new PrintStream(os), new SubProgressMonitor(monitor, 500));
			} finally {
				os.close();
			}

			boolean emptyDiff = false;
			
			if(toClipboard) {				
				final ByteArrayOutputStream baos = (ByteArrayOutputStream)os;
				if(baos.size() == 0) {
					emptyDiff = true;
				} else {
					shell.getDisplay().syncExec(new Runnable() {
						public void run() {
							TextTransfer plainTextTransfer = TextTransfer.getInstance();
							Clipboard clipboard = new Clipboard(shell.getDisplay());		
							clipboard.setContents(
								new String[]{baos.toString()}, 
								new Transfer[]{plainTextTransfer});	
							clipboard.dispose();
						}
					});
				}
			} else {
				if(outputFile.length() == 0) {
					emptyDiff = true;
					outputFile.delete();
				}	
			}

			//check for empty diff and report			
			if (emptyDiff) {
				CVSUIPlugin.openDialog(shell, new CVSUIPlugin.IOpenableInShell() {
					public void open(Shell shell) {
						MessageDialog.openInformation(
							shell,
							Policy.bind("GenerateCVSDiff.noDiffsFoundTitle"), //$NON-NLS-1$
							Policy.bind("GenerateCVSDiff.noDiffsFoundMsg")); //$NON-NLS-1$
					}
				}, CVSUIPlugin.PERFORM_SYNC_EXEC);
			}
		} catch (TeamException e) {
			throw new InvocationTargetException(e);
		} catch(IOException e) {
			throw new InvocationTargetException(e);
		} finally {
			monitor.done();
		}
	}
}
