/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.pipe.resource.memory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.function.LongUnaryOperator;
import org.apache.iotdb.commons.exception.pipe.PipeRuntimeOutOfMemoryCriticalException;
import org.apache.iotdb.commons.memory.IMemoryBlock;
import org.apache.iotdb.commons.memory.MemoryBlockType;
import org.apache.iotdb.commons.pipe.config.PipeConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.pipe.agent.PipeDataNodeAgent;
import org.apache.iotdb.db.pipe.resource.memory.PipeMemoryBlock;
import org.apache.iotdb.db.pipe.resource.memory.PipeMemoryBlockType;
import org.apache.iotdb.db.pipe.resource.memory.PipeModelFixedMemoryBlock;
import org.apache.iotdb.db.pipe.resource.memory.PipeTabletMemoryBlock;
import org.apache.iotdb.db.pipe.resource.memory.PipeTsFileMemoryBlock;
import org.apache.iotdb.db.pipe.resource.memory.strategy.ThresholdAllocationStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipeMemoryManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(PipeMemoryManager.class);
    private static final PipeConfig PIPE_CONFIG = PipeConfig.getInstance();
    private static final boolean PIPE_MEMORY_MANAGEMENT_ENABLED = PipeConfig.getInstance().getPipeMemoryManagementEnabled();
    private final IMemoryBlock memoryBlock = IoTDBDescriptor.getInstance().getMemoryConfig().getPipeMemoryManager().exactAllocate("Stream", MemoryBlockType.DYNAMIC);
    private static final double EXCEED_PROTECT_THRESHOLD = 0.95;
    private volatile long usedMemorySizeInBytesOfTablets;
    private volatile long usedMemorySizeInBytesOfTsFiles;
    private final Set<PipeMemoryBlock> allocatedBlocks = new HashSet<PipeMemoryBlock>();

    public PipeMemoryManager() {
        PipeDataNodeAgent.runtime().registerPeriodicalJob("PipeMemoryManager#tryExpandAll()", this::tryExpandAllAndCheckConsistency, PipeConfig.getInstance().getPipeMemoryExpanderIntervalSeconds());
    }

    private double allowedMaxMemorySizeInBytesOfTabletsAndTsFiles() {
        return (PIPE_CONFIG.getPipeDataStructureTabletMemoryBlockAllocationRejectThreshold() + PIPE_CONFIG.getPipeDataStructureTsFileMemoryBlockAllocationRejectThreshold()) * (double)this.getTotalNonFloatingMemorySizeInBytes();
    }

    private double allowedMaxMemorySizeInBytesOfTablets() {
        return (PIPE_CONFIG.getPipeDataStructureTabletMemoryBlockAllocationRejectThreshold() + PIPE_CONFIG.getPipeDataStructureTsFileMemoryBlockAllocationRejectThreshold() / 2.0) * (double)this.getTotalNonFloatingMemorySizeInBytes();
    }

    private double allowedMaxMemorySizeInBytesOfTsTiles() {
        return (PIPE_CONFIG.getPipeDataStructureTsFileMemoryBlockAllocationRejectThreshold() + PIPE_CONFIG.getPipeDataStructureTabletMemoryBlockAllocationRejectThreshold() / 2.0) * (double)this.getTotalNonFloatingMemorySizeInBytes();
    }

    public boolean isEnough4TabletParsing() {
        return (double)this.usedMemorySizeInBytesOfTablets + (double)this.usedMemorySizeInBytesOfTsFiles < 0.95 * this.allowedMaxMemorySizeInBytesOfTabletsAndTsFiles() && (double)this.usedMemorySizeInBytesOfTablets < 0.95 * this.allowedMaxMemorySizeInBytesOfTablets();
    }

    private boolean isHardEnough4TabletParsing() {
        return (double)this.usedMemorySizeInBytesOfTablets + (double)this.usedMemorySizeInBytesOfTsFiles < this.allowedMaxMemorySizeInBytesOfTabletsAndTsFiles() && (double)this.usedMemorySizeInBytesOfTablets < this.allowedMaxMemorySizeInBytesOfTablets();
    }

    public boolean isEnough4TsFileSlicing() {
        return (double)this.usedMemorySizeInBytesOfTablets + (double)this.usedMemorySizeInBytesOfTsFiles < 0.95 * this.allowedMaxMemorySizeInBytesOfTabletsAndTsFiles() && (double)this.usedMemorySizeInBytesOfTsFiles < 0.95 * this.allowedMaxMemorySizeInBytesOfTsTiles();
    }

    private boolean isHardEnough4TsFileSlicing() {
        return (double)this.usedMemorySizeInBytesOfTablets + (double)this.usedMemorySizeInBytesOfTsFiles < this.allowedMaxMemorySizeInBytesOfTabletsAndTsFiles() && (double)this.usedMemorySizeInBytesOfTsFiles < this.allowedMaxMemorySizeInBytesOfTsTiles();
    }

    public synchronized PipeMemoryBlock forceAllocate(long sizeInBytes) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeMemoryBlock(0L);
        }
        if (sizeInBytes == 0L) {
            return this.registerMemoryBlock(0L);
        }
        return this.forceAllocateWithRetry(sizeInBytes, PipeMemoryBlockType.NORMAL);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PipeTabletMemoryBlock forceAllocateForTabletWithRetry(long tabletSizeInBytes) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeTabletMemoryBlock(0L);
        }
        if (tabletSizeInBytes == 0L) {
            return (PipeTabletMemoryBlock)this.registerMemoryBlock(0L, PipeMemoryBlockType.TABLET);
        }
        int size = PIPE_CONFIG.getPipeMemoryAllocateMaxRetries();
        for (int i = 1; i <= size && !this.isHardEnough4TabletParsing(); ++i) {
            try {
                Thread.sleep(PIPE_CONFIG.getPipeMemoryAllocateRetryIntervalInMs());
                continue;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocateWithRetry: interrupted while waiting for available memory", (Throwable)ex);
            }
        }
        if (!this.isHardEnough4TabletParsing()) {
            throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocateForTablet: failed to allocate because there's too much memory for tablets, total memory size %d bytes, used memory for tablet size %d bytes, requested memory size %d bytes", this.getTotalNonFloatingMemorySizeInBytes(), this.usedMemorySizeInBytesOfTablets, tabletSizeInBytes));
        }
        PipeMemoryManager pipeMemoryManager = this;
        synchronized (pipeMemoryManager) {
            PipeTabletMemoryBlock block = (PipeTabletMemoryBlock)this.forceAllocateWithRetry(tabletSizeInBytes, PipeMemoryBlockType.TABLET);
            this.usedMemorySizeInBytesOfTablets += block.getMemoryUsageInBytes();
            return block;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PipeTsFileMemoryBlock forceAllocateForTsFileWithRetry(long tsFileSizeInBytes) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeTsFileMemoryBlock(0L);
        }
        if (tsFileSizeInBytes == 0L) {
            return (PipeTsFileMemoryBlock)this.registerMemoryBlock(0L, PipeMemoryBlockType.TS_FILE);
        }
        int size = PIPE_CONFIG.getPipeMemoryAllocateMaxRetries();
        for (int i = 1; i <= size && !this.isHardEnough4TsFileSlicing(); ++i) {
            try {
                Thread.sleep(PIPE_CONFIG.getPipeMemoryAllocateRetryIntervalInMs());
                continue;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocateWithRetry: interrupted while waiting for available memory", (Throwable)ex);
            }
        }
        if (!this.isHardEnough4TsFileSlicing()) {
            throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocateForTsFile: failed to allocate because there's too much memory for tsfiles, total memory size %d bytes, used memory for tsfile size %d bytes, requested memory size %d bytes", this.getTotalNonFloatingMemorySizeInBytes(), this.usedMemorySizeInBytesOfTsFiles, tsFileSizeInBytes));
        }
        PipeMemoryManager pipeMemoryManager = this;
        synchronized (pipeMemoryManager) {
            PipeTsFileMemoryBlock block = (PipeTsFileMemoryBlock)this.forceAllocateWithRetry(tsFileSizeInBytes, PipeMemoryBlockType.TS_FILE);
            this.usedMemorySizeInBytesOfTsFiles += block.getMemoryUsageInBytes();
            return block;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PipeModelFixedMemoryBlock forceAllocateForModelFixedMemoryBlock(long fixedSizeInBytes, PipeMemoryBlockType type) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeModelFixedMemoryBlock(Long.MAX_VALUE, new ThresholdAllocationStrategy());
        }
        if (fixedSizeInBytes == 0L) {
            return (PipeModelFixedMemoryBlock)this.registerMemoryBlock(0L, type);
        }
        int size = PIPE_CONFIG.getPipeMemoryAllocateMaxRetries();
        for (int i = 1; i <= size && this.getFreeMemorySizeInBytes() < fixedSizeInBytes; ++i) {
            try {
                Thread.sleep(PIPE_CONFIG.getPipeMemoryAllocateRetryIntervalInMs());
                continue;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocateWithRetry: interrupted while waiting for available memory", (Throwable)ex);
            }
        }
        PipeMemoryManager pipeMemoryManager = this;
        synchronized (pipeMemoryManager) {
            if (this.getFreeMemorySizeInBytes() < fixedSizeInBytes) {
                return (PipeModelFixedMemoryBlock)this.forceAllocateWithRetry(this.getFreeMemorySizeInBytes(), type);
            }
            return (PipeModelFixedMemoryBlock)this.forceAllocateWithRetry(fixedSizeInBytes, type);
        }
    }

    private PipeMemoryBlock forceAllocateWithRetry(long sizeInBytes, PipeMemoryBlockType type) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            switch (type) {
                case TABLET: {
                    return new PipeTabletMemoryBlock(sizeInBytes);
                }
                case TS_FILE: {
                    return new PipeTsFileMemoryBlock(sizeInBytes);
                }
                case BATCH: 
                case WAL: {
                    return new PipeModelFixedMemoryBlock(sizeInBytes, new ThresholdAllocationStrategy());
                }
            }
            return new PipeMemoryBlock(sizeInBytes);
        }
        int memoryAllocateMaxRetries = PIPE_CONFIG.getPipeMemoryAllocateMaxRetries();
        for (int i = 1; i <= memoryAllocateMaxRetries; ++i) {
            if (this.getTotalNonFloatingMemorySizeInBytes() - this.memoryBlock.getUsedMemoryInBytes() >= sizeInBytes) {
                return this.registerMemoryBlock(sizeInBytes, type);
            }
            try {
                this.tryShrinkUntilFreeMemorySatisfy(sizeInBytes);
                this.wait(PIPE_CONFIG.getPipeMemoryAllocateRetryIntervalInMs());
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocate: interrupted while waiting for available memory", (Throwable)e);
            }
        }
        throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocate: failed to allocate memory after %d retries, total memory size %d bytes, used memory size %d bytes, requested memory size %d bytes", memoryAllocateMaxRetries, this.getTotalNonFloatingMemorySizeInBytes(), this.memoryBlock.getUsedMemoryInBytes(), sizeInBytes));
    }

    public synchronized void forceResize(PipeMemoryBlock block, long targetSize) {
        if (block == null || block.isReleased()) {
            LOGGER.warn("forceResize: cannot resize a null or released memory block");
            return;
        }
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            block.setMemoryUsageInBytes(targetSize);
            return;
        }
        long oldSize = block.getMemoryUsageInBytes();
        if (oldSize == 0L) {
            this.allocatedBlocks.add(block);
        }
        if (oldSize >= targetSize) {
            this.memoryBlock.release(oldSize - targetSize);
            if (block instanceof PipeTabletMemoryBlock) {
                this.usedMemorySizeInBytesOfTablets -= oldSize - targetSize;
            }
            if (block instanceof PipeTsFileMemoryBlock) {
                this.usedMemorySizeInBytesOfTsFiles -= oldSize - targetSize;
            }
            block.setMemoryUsageInBytes(targetSize);
            if (targetSize == 0L) {
                this.allocatedBlocks.remove(block);
            }
            return;
        }
        long sizeInBytes = targetSize - oldSize;
        int memoryAllocateMaxRetries = PIPE_CONFIG.getPipeMemoryAllocateMaxRetries();
        for (int i = 1; i <= memoryAllocateMaxRetries; ++i) {
            if (this.getTotalNonFloatingMemorySizeInBytes() - this.memoryBlock.getUsedMemoryInBytes() >= sizeInBytes) {
                this.memoryBlock.forceAllocateWithoutLimitation(sizeInBytes);
                if (block instanceof PipeTabletMemoryBlock) {
                    this.usedMemorySizeInBytesOfTablets += sizeInBytes;
                }
                if (block instanceof PipeTsFileMemoryBlock) {
                    this.usedMemorySizeInBytesOfTsFiles += sizeInBytes;
                }
                block.setMemoryUsageInBytes(targetSize);
                return;
            }
            try {
                this.tryShrinkUntilFreeMemorySatisfy(sizeInBytes);
                this.wait(PIPE_CONFIG.getPipeMemoryAllocateRetryIntervalInMs());
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceResize: interrupted while waiting for available memory", (Throwable)e);
            }
        }
        throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceResize: failed to allocate memory after %d retries, total memory size %d bytes, used memory size %d bytes, requested memory size %d bytes", memoryAllocateMaxRetries, this.getTotalNonFloatingMemorySizeInBytes(), this.memoryBlock.getUsedMemoryInBytes(), sizeInBytes));
    }

    public synchronized PipeMemoryBlock forceAllocateIfSufficient(long sizeInBytes, float usedThreshold) {
        if (usedThreshold < 0.0f || usedThreshold > 1.0f) {
            return null;
        }
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeMemoryBlock(sizeInBytes);
        }
        if (sizeInBytes == 0L) {
            return this.registerMemoryBlock(0L);
        }
        if ((float)(this.memoryBlock.getUsedMemoryInBytes() + sizeInBytes) <= (float)this.getTotalNonFloatingMemorySizeInBytes() * usedThreshold) {
            return this.forceAllocate(sizeInBytes);
        }
        return null;
    }

    public synchronized PipeMemoryBlock tryAllocate(long sizeInBytes) {
        return this.tryAllocate(sizeInBytes, currentSize -> currentSize * 2L / 3L);
    }

    public synchronized PipeMemoryBlock tryAllocate(long sizeInBytes, LongUnaryOperator customAllocateStrategy) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeMemoryBlock(sizeInBytes);
        }
        if (sizeInBytes == 0L || this.getTotalNonFloatingMemorySizeInBytes() - this.memoryBlock.getUsedMemoryInBytes() >= sizeInBytes) {
            return this.registerMemoryBlock(sizeInBytes);
        }
        long sizeToAllocateInBytes = sizeInBytes;
        long memoryAllocateMinSizeInBytes = PIPE_CONFIG.getPipeMemoryAllocateMinSizeInBytes();
        while (sizeToAllocateInBytes > memoryAllocateMinSizeInBytes) {
            if (this.getTotalNonFloatingMemorySizeInBytes() - this.memoryBlock.getUsedMemoryInBytes() >= sizeToAllocateInBytes) {
                LOGGER.info("tryAllocate: allocated memory, total memory size {} bytes, used memory size {} bytes, original requested memory size {} bytes, actual requested memory size {} bytes", new Object[]{this.getTotalNonFloatingMemorySizeInBytes(), this.memoryBlock.getUsedMemoryInBytes(), sizeInBytes, sizeToAllocateInBytes});
                return this.registerMemoryBlock(sizeToAllocateInBytes);
            }
            sizeToAllocateInBytes = Math.max(customAllocateStrategy.applyAsLong(sizeToAllocateInBytes), memoryAllocateMinSizeInBytes);
        }
        if (this.tryShrinkUntilFreeMemorySatisfy(sizeToAllocateInBytes)) {
            LOGGER.info("tryAllocate: allocated memory, total memory size {} bytes, used memory size {} bytes, original requested memory size {} bytes, actual requested memory size {} bytes", new Object[]{this.getTotalNonFloatingMemorySizeInBytes(), this.memoryBlock.getUsedMemoryInBytes(), sizeInBytes, sizeToAllocateInBytes});
            return this.registerMemoryBlock(sizeToAllocateInBytes);
        }
        LOGGER.warn("tryAllocate: failed to allocate memory, total memory size {} bytes, used memory size {} bytes, requested memory size {} bytes", new Object[]{this.getTotalNonFloatingMemorySizeInBytes(), this.memoryBlock.getUsedMemoryInBytes(), sizeInBytes});
        return this.registerMemoryBlock(0L);
    }

    public synchronized boolean tryAllocate(PipeMemoryBlock block, long memoryInBytesNeededToBeAllocated) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return false;
        }
        if (this.getTotalNonFloatingMemorySizeInBytes() - this.memoryBlock.getUsedMemoryInBytes() >= memoryInBytesNeededToBeAllocated) {
            this.memoryBlock.forceAllocateWithoutLimitation(memoryInBytesNeededToBeAllocated);
            if (block instanceof PipeTabletMemoryBlock) {
                this.usedMemorySizeInBytesOfTablets += memoryInBytesNeededToBeAllocated;
            }
            if (block instanceof PipeTsFileMemoryBlock) {
                this.usedMemorySizeInBytesOfTsFiles += memoryInBytesNeededToBeAllocated;
            }
            block.setMemoryUsageInBytes(block.getMemoryUsageInBytes() + memoryInBytesNeededToBeAllocated);
            return true;
        }
        return false;
    }

    private PipeMemoryBlock registerMemoryBlock(long sizeInBytes) {
        return this.registerMemoryBlock(sizeInBytes, PipeMemoryBlockType.NORMAL);
    }

    private PipeMemoryBlock registerMemoryBlock(long sizeInBytes, PipeMemoryBlockType type) {
        PipeMemoryBlock returnedMemoryBlock;
        switch (type) {
            case TABLET: {
                returnedMemoryBlock = new PipeTabletMemoryBlock(sizeInBytes);
                break;
            }
            case TS_FILE: {
                returnedMemoryBlock = new PipeTsFileMemoryBlock(sizeInBytes);
                break;
            }
            case BATCH: 
            case WAL: {
                returnedMemoryBlock = new PipeModelFixedMemoryBlock(sizeInBytes, new ThresholdAllocationStrategy());
                break;
            }
            default: {
                returnedMemoryBlock = new PipeMemoryBlock(sizeInBytes);
            }
        }
        if (sizeInBytes > 0L) {
            this.memoryBlock.forceAllocateWithoutLimitation(sizeInBytes);
            this.allocatedBlocks.add(returnedMemoryBlock);
        }
        return returnedMemoryBlock;
    }

    private boolean tryShrinkUntilFreeMemorySatisfy(long sizeInBytes) {
        boolean hasAtLeastOneBlockShrinkable;
        ArrayList<PipeMemoryBlock> shuffledBlocks = new ArrayList<PipeMemoryBlock>(this.allocatedBlocks);
        Collections.shuffle(shuffledBlocks);
        do {
            hasAtLeastOneBlockShrinkable = false;
            for (PipeMemoryBlock block : shuffledBlocks) {
                if (!block.shrink()) continue;
                hasAtLeastOneBlockShrinkable = true;
                if (this.getTotalNonFloatingMemorySizeInBytes() - this.memoryBlock.getUsedMemoryInBytes() < sizeInBytes) continue;
                return true;
            }
        } while (hasAtLeastOneBlockShrinkable);
        return false;
    }

    public synchronized void tryExpandAllAndCheckConsistency() {
        this.allocatedBlocks.forEach(PipeMemoryBlock::expand);
        long blockSum = this.allocatedBlocks.stream().mapToLong(PipeMemoryBlock::getMemoryUsageInBytes).sum();
        if (blockSum != this.memoryBlock.getUsedMemoryInBytes()) {
            LOGGER.warn("tryExpandAllAndCheckConsistency: memory usage is not consistent with allocated blocks, usedMemorySizeInBytes is {} but sum of all blocks is {}", (Object)this.memoryBlock.getUsedMemoryInBytes(), (Object)blockSum);
        }
        long tabletBlockSum = this.allocatedBlocks.stream().filter(PipeTabletMemoryBlock.class::isInstance).mapToLong(PipeMemoryBlock::getMemoryUsageInBytes).sum();
        if (tabletBlockSum != this.usedMemorySizeInBytesOfTablets) {
            LOGGER.warn("tryExpandAllAndCheckConsistency: memory usage of tablets is not consistent with allocated blocks, usedMemorySizeInBytesOfTablets is {} but sum of all tablet blocks is {}", (Object)this.usedMemorySizeInBytesOfTablets, (Object)tabletBlockSum);
        }
        long tsFileBlockSum = this.allocatedBlocks.stream().filter(PipeTsFileMemoryBlock.class::isInstance).mapToLong(PipeMemoryBlock::getMemoryUsageInBytes).sum();
        if (tsFileBlockSum != this.usedMemorySizeInBytesOfTsFiles) {
            LOGGER.warn("tryExpandAllAndCheckConsistency: memory usage of tsfiles is not consistent with allocated blocks, usedMemorySizeInBytesOfTsFiles is {} but sum of all tsfile blocks is {}", (Object)this.usedMemorySizeInBytesOfTsFiles, (Object)tsFileBlockSum);
        }
    }

    public synchronized void release(PipeMemoryBlock block) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return;
        }
        this.allocatedBlocks.remove(block);
        this.memoryBlock.release(block.getMemoryUsageInBytes());
        if (block instanceof PipeTabletMemoryBlock) {
            this.usedMemorySizeInBytesOfTablets -= block.getMemoryUsageInBytes();
        }
        if (block instanceof PipeTsFileMemoryBlock) {
            this.usedMemorySizeInBytesOfTsFiles -= block.getMemoryUsageInBytes();
        }
        block.markAsReleased();
        this.notifyAll();
    }

    public synchronized boolean release(PipeMemoryBlock block, long sizeInBytes) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return false;
        }
        this.memoryBlock.release(sizeInBytes);
        if (block instanceof PipeTabletMemoryBlock) {
            this.usedMemorySizeInBytesOfTablets -= sizeInBytes;
        }
        if (block instanceof PipeTsFileMemoryBlock) {
            this.usedMemorySizeInBytesOfTsFiles -= sizeInBytes;
        }
        block.setMemoryUsageInBytes(block.getMemoryUsageInBytes() - sizeInBytes);
        this.notifyAll();
        return true;
    }

    public long getUsedMemorySizeInBytes() {
        return this.memoryBlock.getUsedMemoryInBytes();
    }

    public long getUsedMemorySizeInBytesOfTablets() {
        return this.usedMemorySizeInBytesOfTablets;
    }

    public long getUsedMemorySizeInBytesOfTsFiles() {
        return this.usedMemorySizeInBytesOfTsFiles;
    }

    public long getAllocatedMemorySizeInBytesOfBatch() {
        return (long)(PipeConfig.getInstance().getPipeDataStructureBatchMemoryProportion() * (double)this.getTotalNonFloatingMemorySizeInBytes());
    }

    public long getFreeMemorySizeInBytes() {
        return this.memoryBlock.getFreeMemoryInBytes();
    }

    public long getTotalNonFloatingMemorySizeInBytes() {
        return (long)((double)this.memoryBlock.getTotalMemorySizeInBytes() * (1.0 - PipeConfig.getInstance().getPipeTotalFloatingMemoryProportion()));
    }

    public long getTotalFloatingMemorySizeInBytes() {
        return (long)((double)this.memoryBlock.getTotalMemorySizeInBytes() * PipeConfig.getInstance().getPipeTotalFloatingMemoryProportion());
    }

    public long getTotalMemorySizeInBytes() {
        return this.memoryBlock.getTotalMemorySizeInBytes();
    }
}

